// Design: calculator
// Description: Simple calculator for Digilent's Basys 2 board
// Author: Jorge Juan <jjchico@dte.us.es>
// Copyright Universidad de Sevilla, Spain
// Date: 13-11-2011 (initial version)

//////////////////////////////////////////////////////////////////////////
// Calculator                                                           //
//////////////////////////////////////////////////////////////////////////

/*
   This design describes a simple hexadecimal calculator. The calculator does
   four simple operations based on the 4-bit operation selection input signal
   'op':

    op  | result
   -----+-------
   1000 | a + b
   0100 | a - b
   0010 | b - a
   0001 | a * b

   Operands 'a', 'b' are 4 bits (one hexadecimal digit). Result is 8 bits (two
   hexadecimal digits).

   Output is generated by a 7-segment display controller intended to drive a
   4 digit hexadecimal 7-segment display as the one found in Digilent's FPGA
   boards.
 */

module calculator #(
    parameter cdbits = 18   // clock divider bits (for disp. controller)
    )(
    input wire ck,          // system clock (display controller)
    input wire [3:0] a,     // first operand
    input wire [3:0] b,     // second operand
    input wire [3:0] op,    // decoded operation selector
    output wire [0:6] seg,  // 7-segment output
    output wire [3:0] an,   // anode output
    output wire dp          // decimal point output (disp. cont.)
    );

    // Internal signals
    wire [7:0] z;           // calculation result
    wire [3:0] dp_in;       // decimal point connection

    // Arithmetic unit
    arithmetic_unit arithm1 (.op(op), .a(a), .b(b), .z(z));

    // Decimal points are all off
    assign dp_in = 4'b1111;

    // Display controller instantiation
    display_ctrl #(.cdbits(cdbits)) dc1 (
        .ck(ck), .x3(a), .x2(b), .x1(z[7:4]), .x0(z[3:0]),
        .dp_in(dp_in), .seg(seg), .an(an), .dp(dp)
    );

endmodule // calculator
